% This script numerically solves the equilibration of a 2 drop system
% in which each drop has an open interface on top and bottom
clear all;

%% SIMULATION SETUP
% ---------------
% Dimensions of the device
% ---------------
Rin   = 2e-3     ; 
Rout  = 2.25e-3  ;
h     = 1.2e-3   ;
wch   = 0.8e-3   ;
hch   = 0.8e-3   ;
lch   = 2e-3     ;
eta   = 1e-3     ;
glam  = (1+wch/hch)^2/(wch^2/hch^2)          ;
Rflu  = 8 * eta * lch * glam / (wch^3 * hch) ;

% ---------------
% Physical constants
% ---------------
global rho g gamma ;
gamma = 0.07       ;
rho   = 1000       ;
g     = 9.8      ;   

% ---------------
% Usesr 
% ---------------
Vini  = 50e-9    ;
Vbol  = 15e-9    ;
Vrem  = 15e-9    ;

% ---------------
% Calculated volumes
% ---------------
Vc      = lch * wch * hch     ; % volume of the channel
Vin     = h * pi * Rin^2      ; % volume of the port on the input side
Vout    = h * pi * Rout^2     ; % volume of the port on thr output side

% ---------------
% Simulation starting variables
% ---------------
Ratio   = Vin / (Vout + Vin)  ; % Ratio of the volumes of both sides
V1      = Vini * Ratio        ; % starting input volume
V2      = Vini * (1 - Ratio)  ; % starting output volume

% ---------------
% Simulation parameters
% ---------------
simEnd  = 1e-1     ; % Threshold at which the simulation stops
dt      = 0.001      ; % Factor to calculate volume exchange at each step
log     = true     ; % flag for debugging

% ---------------
% Setup arrays to plot the evolution of the radii
% ---------------
n       = 10000           ; % number of points
indeces = linspace(0,n,n) ; % Indexes list

% ---------------
% Get the pressure - volume relationship for the drops
% ---------------

% Make the pressure - volume table for the input drop
PVHS_bot_in  = Function_DropPressureVolume_V03( Rin, 20, -1, false )     ;
display('Determined pressure - volume tables for input pendant drops')  ;

% Make the pressure - volume table for the input drop
PVHS_top_in  = Function_DropPressureVolume_V03( Rin, 20, 1, false )    ;
display('Determined pressure - volume tables for input top drops')      ;

% Make the pressure - volume table for the input drop
PVHS_bot_out = Function_DropPressureVolume_V03( Rout, 20, -1, false )    ;
display('Determined pressure - volume tables for output pendant drops') ;

% Make the pressure - volume table for the input drop
PVHS_top_out = Function_DropPressureVolume_V03( Rout, 20, 1, false )   ;
display('Determined pressure - volume tables for output top drops') ;


%% PRESIMULATION EQUILIBRATION
% ---------------
% Equilibrate the 2 drops with the initial volume
% ---------------
[ ~, ~, V1, V2 ] = Sub_Equilibrate_All_V03( V1, Rin, PVHS_bot_in, PVHS_top_in, V2, Rout, PVHS_bot_out, PVHS_top_out, Rflu, h, simEnd, dt, n, log);

%% SIMULATION RUN
% Add the additional bolus in the input port
V1m  = V1 * ones(1,500)  ;
V2m  = V2 * ones(1,500)  ;
V1   = V1 + Vbol         ;
%V2   = V2 + Vbol         ;

% ---------------
% Equilibrate the 2 drops 
% ---------------
[ DropIn, DropOut, V1, V2 ] = Sub_Equilibrate_All_V03( V1, Rin, PVHS_bot_in, PVHS_top_in, V2, Rout, PVHS_bot_out, PVHS_top_out, Rflu, h, simEnd, dt, n, log);
display('End of droplet equilibration')                       ;
V1m   = [ V1m DropIn.volume(1:DropIn.endIndex) ]              ;
V2m   = [ V2m DropOut.volume(1:DropOut.endIndex) ]            ; 
V1m   = [ V1m DropIn.volume(DropIn.endIndex)*ones(1,500) ]    ;
V2m   = [ V2m DropOut.volume(DropOut.endIndex)*ones(1,500) ]  ;

% Remove the additional bolus in the input port
V1   = V1 + Vbol         ;

% ---------------
% Equilibrate the 2 drops 
% ---------------
[ DropIn, DropOut, V1, V2 ] = Sub_Equilibrate_All_V03( V1, Rin, PVHS_bot_in, PVHS_top_in, V2, Rout, PVHS_bot_out, PVHS_top_out, Rflu, h, simEnd, dt, n, log);
display('End of droplet equilibration')            ;
V1m   = [ V1m DropIn.volume(1:DropIn.endIndex) ]   ;
V2m   = [ V2m DropOut.volume(1:DropOut.endIndex) ] ;
V1m   = [ V1m DropIn.volume(DropIn.endIndex)*ones(1,500) ]    ;
V2m   = [ V2m DropOut.volume(DropOut.endIndex)*ones(1,500) ]  ;

% Remove the additional bolus in the input port
V1   = V1 - Vrem ;

% ---------------
% Equilibrate the 2 drops 
% ---------------
[ DropIn, DropOut, V1, V2 ] = Sub_Equilibrate_All_V03( V1, Rin, PVHS_bot_in, PVHS_top_in, V2, Rout, PVHS_bot_out, PVHS_top_out, Rflu, h, simEnd, dt, n, log);
display('End of droplet equilibration')            ;
V1m   = [ V1m DropIn.volume(1:DropIn.endIndex) ]   ;
V2m   = [ V2m DropOut.volume(1:DropOut.endIndex) ] ;
V1m   = [ V1m DropIn.volume(DropIn.endIndex)*ones(1,500) ]    ;
V2m   = [ V2m DropOut.volume(DropOut.endIndex)*ones(1,500) ]  ;

% Remove the additional bolus in the input port
V1   = V1 - Vrem ;

% ---------------
% Equilibrate the 2 drops 
% ---------------
[ DropIn, DropOut, V1, V2 ] = Sub_Equilibrate_All_V03( V1, Rin, PVHS_bot_in, PVHS_top_in, V2, Rout, PVHS_bot_out, PVHS_top_out, Rflu, h, simEnd, dt, n, log);
display('End of droplet equilibration')            ;
V1m   = [ V1m DropIn.volume(1:DropIn.endIndex) ]   ;
V2m   = [ V2m DropOut.volume(1:DropOut.endIndex) ] ;
V1m   = [ V1m DropIn.volume(DropIn.endIndex)*ones(1,500) ]    ;
V2m   = [ V2m DropOut.volume(DropOut.endIndex)*ones(1,500) ]  ;

%% PLOTTING

% ---------------
% Plots
% ---------------
figure(5)
plot(V1m,'b')    ; 
hold on                                                                  ;
plot(V2m,'r') ; 
xlabel('Simulation time')                                                ; 
ylabel('Volume')                                                         ; 
title('Volume of drops')                                                 ; 

%figure(6)
%plot(PVHS_bot_in.volumes, PVHS_bot_in.pressures,'b')                     ; 
%hold on                                                                  ;
%plot(PVHS_top_in.volumes, PVHS_top_in.pressures,'b-')                    ; 
%hold on                                                                  ;
%plot(PVHS_bot_out.volumes, PVHS_bot_out.pressures,'r')                   ; 
%hold on                                                                  ;
%plot(PVHS_top_out.volumes, PVHS_top_out.pressures,'r-')                  ; 
%xlabel('Volumes of pendant drop')                                        ; 
%ylabel('Pressure')                                                       ; 
%title('Pressures in function of volume')                                 ; 

% ---------------
% Graphics
% ---------------
[ DropLineXIntop, DropLineYIntop , DeviceLineXIntop, DeviceLineYIntop ]     = Graphics_PlotSuspendedDrops_V03( Rin, DropIn.topheight(DropIn.endIndex), h/2, log ) ;
[ DropLineXInbot, DropLineYInbot , DeviceLineXInbot, DeviceLineYInbot ]     = Graphics_PlotSuspendedDrops_V03( Rin, DropIn.botheight(DropIn.endIndex), h/2, log ) ;
[ DropLineXOuttop, DropLineYOuttop , DeviceLineXOuttop, DeviceLineYOuttop ] = Graphics_PlotSuspendedDrops_V03( Rout, DropOut.topheight(DropOut.endIndex), h/2, log ) ;
[ DropLineXOutbot, DropLineYOutbot , DeviceLineXOutbot, DeviceLineYOutbot ] = Graphics_PlotSuspendedDrops_V03( Rout, DropOut.botheight(DropOut.endIndex), h/2, log ) ;

% make the figure and legends
f1 = figure(1)                 ;
clf(f1)                        ;
xlabel('x (mm)')               ; 
ylabel('y (mm)')               ;

% draw the outlines of the input pendant drop 
scatter(DropLineXInbot,DropLineYInbot,'b')            ; 
hold on                                               ;
scatter(DeviceLineXInbot,DeviceLineYInbot,'black')    ; 
hold on                                               ;

% draw the outlines of the input top drop 
scatter(DropLineXIntop,-DropLineYIntop,'cyan')           ; 
hold on                                               ;
scatter(DeviceLineXIntop,-DeviceLineYIntop,'black')   ; 
hold on                                               ;

% draw the outlines of the output pendant drop 
scatter(-DropLineXOutbot,DropLineYOutbot,'b')         ; 
hold on                                               ;
scatter(-DeviceLineXOutbot,DeviceLineYOutbot,'black') ; 
hold on                                               ;

% draw the outlines of the output top drop 
scatter(-DropLineXOuttop,-DropLineYOuttop,'cyan')        ; 
hold on                                               ;
scatter(-DeviceLineXOuttop,-DeviceLineYOuttop,'black'); 
hold on                                               ;

% % ---------------
% % Movie
% % ---------------
% windowSize  = 1000             ;
% Raverage    = (Rin+Rout) / 2   ;
% Rmax        = max(Rin, Rout)   ;
% PixSize     = 2*(Rin+Rout)/windowSize ;
% 
% mov = figure('Position',[100 100 850 600])   ;
% frames            = linspace(1,9001,451);
% %frames            = [1 51 101 151 201 251 301 351 401 451 501 551 601 651 701 751 801 851 901 951 1001 1501 2001 2501 3001 3501 4001 4501 5001 5501 6001 6501 7001 7501 8001 8501 9501 9999];
% M(length(frames)) = struct('cdata',[],'colormap',[])   ;
% 
% for index=1:length(frames)
%     i = frames(index) ;
%     clf(mov);
%     xlabel('x (mm)')                ; 
%     ylabel('y (mm)')                ;
%     
%     % Get the outlines of the geometry and the drop
%     [outlineX, outlineY, dropX, dropY] = Plot_DropsV03( R11m(i), H11m(i), Rin, h/2, windowSize, PixSize, 2*Rmax, 2*Rmax, 1 ) ;
%     [outline2X, outline2Y, drop2X, drop2Y] = Plot_DropsV03( R12m(i), H12m(i), Rin, h/2, windowSize, PixSize, 2*Rmax, 2*Rmax, -1 ) ;
% 
%     % draw the outlines of the geometry
%     scatter(outlineX(1:windowSize),outlineY(1:windowSize),'black')        ; 
%     hold on                                                               ;
%     scatter(dropX(1:windowSize),dropY(1:windowSize),'b')                  ; 
%     hold on                                                               ;
%     
%     % draw the drop outline
%     scatter(outline2X(1:windowSize),outline2Y(1:windowSize),'black')      ; 
%     hold on                                                               ;
%     scatter(drop2X(1:windowSize),drop2Y(1:windowSize),'b')                ; 
%     hold on                                                               ;
%     
%     % make the frame and draw it
%     frameX = [-2*Rmax -2*Rmax 2*Rmax 2*Rmax] ;
%     frameY = [-2*Rmax 2*Rmax -2*Rmax 2*Rmax] ;
%     scatter(frameX,frameY,'black')           ; 
% 
%     M(index) = getframe(mov)   ;
% end
% 
% clf(mov);
% movie(mov,M) ;
% 
% writerObj = VideoWriter('C:\Users\Erwin\Personal\erwin.berthier@gmail.com\PROJECT\14-07-19 - Suspended drop cultures\Simulation\MovieIn','MPEG-4') ; 
% open(writerObj) ;
% writeVideo(writerObj,M) ;
% close(writerObj) ;
% 
% 
% % ---------------
% % Movie
% % ---------------
% mov2 = figure('Position',[100 100 850 600])   ;
% M2(length(frames)) = struct('cdata',[],'colormap',[])   ;
% 
% for index=1:length(frames)
%     i = frames(index) ;
%     clf(mov2);
%     xlabel('x (mm)')                ; 
%     ylabel('y (mm)')                ;
%     axis manual                     ;
%     axis([-2*Rmax 2*Rmax -2*Rmax 2*Rmax]) ;
%     
%     % Get the outlines of the geometry and the drop
%     [outlineX, outlineY, dropX, dropY] = Plot_DropsV03( R21m(i), H21m(i), Rout, h/2, windowSize, PixSize, 2*Rmax, 2*Rmax, 1 ) ;
%     [outline2X, outline2Y, drop2X, drop2Y] = Plot_DropsV03( R22m(i), H22m(i), Rout, h/2, windowSize, PixSize, 2*Rmax, 2*Rmax, -1 ) ;
% 
%     % draw the outlines of the geometry
%     scatter(outlineX(1:windowSize),outlineY(1:windowSize),'black')        ; 
%     hold on                                                               ;
%     scatter(dropX(1:windowSize),dropY(1:windowSize),'b')                  ; 
%     hold on                                                               ;
%     
%     % draw the drop outline
%     scatter(outline2X(1:windowSize),outline2Y(1:windowSize),'black')      ; 
%     hold on                                                               ;
%     scatter(drop2X(1:windowSize),drop2Y(1:windowSize),'b')                ; 
%     hold on                                                               ;
%     
%     % make the frame and draw it
%     frameX = [-2*Rmax -2*Rmax 2*Rmax 2*Rmax] ;
%     frameY = [-2*Rmax 2*Rmax -2*Rmax 2*Rmax] ;
%     scatter(frameX,frameY,'black')           ;
% 
%     M2(index) = getframe(mov2)   ;
% end
% 
% clf(mov2);
% movie(mov2,M2) ;
% 
% writerObj2 = VideoWriter('C:\Users\Erwin\Personal\erwin.berthier@gmail.com\PROJECT\14-07-19 - Suspended drop cultures\Simulation\MovieOut','MPEG-4') ; 
% open(writerObj2) ;
% writeVideo(writerObj2,M2) ;
% close(writerObj2) ;


